<?php
namespace controller;

use \lib\db as db;
use \lib\Form as Form;
use \lib\Lang as Lang;

Lang::load('pawn');
Lang::load('admin');

class Pawn extends \Pawn\Controller
{
	# Database setup
	public static function setup()
	{
		# Check compatibility
		$version = phpversion();
		
		if(version_compare(phpversion(), '5.3.0', '<'))
		{
			throw new \Pawn\Exception('Configuration', 'You must be running PHP 5.3 or above to run this script, your PHP version is '.$version.'.', false, false);
		}
		
		$drivers = \PDO::getAvailableDrivers();
		
		if(!count($drivers))
		{
			throw new \Pawn\Exception('Configuration', 'There are no enabled PDO driver extensions, try to enable on in your PHP.ini file.', false, false);
		}
		
		if(!is_writable(ROOT.'cache'))
		{
			throw new \Pawn\Exception('Configuration', 'The <b>/cache</b> folder is not writable. Try chmodding it to 775 or 777 and refresh the page to see if the error disappears.', false, false);
		}
		
		if(!is_writable(ROOT.'public/db'))
		{
			throw new \Pawn\Exception('Configuration', 'The <b>/public/db</b> folder is not writable. Try chmodding it to 775 or 777 and refresh the page to see if the error disappears.', false, false);
		}
		
		# Database settings
		$form = new Form('setup');
		
		$form->field('driver', 'select', $drivers);
		$form->field('host');
		$form->field('username');
		$form->field('password', 'password', array('optional'=>true));
		$form->field('database');
		$form->field('prefix', 'text', array('optional'=>true, 'max_length'=>30), str_replace(' ', '_', strtolower(\Pawn::setting('title'))).'_');
		
		if($form->validate())
		{
			$valid = true;
			
			try
			{
				$handler = new \PDO($form->driver.':host='.$form->host.';dbname='.$form->database, $form->username, $form->password);
				$handler->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);
			} catch(\Exception $e)
			{
				$valid = false;
				
				switch($e->getCode())
				{
					case 2002:
						$form->host->error('Host not found.');
						break;
					case 1045:
						$form->username->error('Access to database not allowed.');
						$form->password->error('The password might be wrong.');
						break;
					case 1049:
						$form->database->error('No such database.');
						break;
					case 1044:
						$form->username->error("User doesn't have permission to access database.");
						break;
				}
			}
			
			if($valid)
			{
				db::setHandler($handler);
				
				db::execute('CREATE TABLE IF NOT EXISTS `'.$form->prefix.'config` (
	`key`   VARCHAR(32) PRIMARY KEY,
	`value` TEXT
)', true);
				
				$query = '';
				$settings = \Pawn::settings();
				
				foreach($settings as $key=>$value)
				{
					$query .= ', ('.db::quote($key).', '.db::quote($value).')';
				}
				
				db::execute('INSERT INTO `'.$form->prefix.'config` VALUES ("version", "0.0.0"), ("setup", '.time().')'.$query);
				
				db::execute('CREATE TABLE IF NOT EXISTS `'.$form->prefix.'migration` (
	`model`   VARCHAR(32) PRIMARY KEY,
	`version` TINYINT UNSIGNED
)', true);

				$form->hide();
				
				# Save app.php
				return "<?php\nreturn array (\n  'driver' => '$form->driver',\n  'host' => '$form->host',\n  'username' => '$form->username',\n  'password' => '$form->password',\n  'database' => '$form->database',\n  'prefix' => '$form->prefix',\n  'secret' => '".md5(uniqid())."' # Do not modify this!\n);";
			}
		}
		
		\lib\Template::render('pawn/setup.html', array
		(
			'heading'		=>	Lang::get('heading_setup'),
			'about'			=>	Lang::get('database'),
			'sidebar_about'	=>	Lang::get('setup_process'),
			
			'form'		=>	$form
		));
		
		# Don't save app.php
		exit;
	}
	
	# Super-user creation
	public static function super_user()
	{
		$user = new \model\User;
		
		$form = $user->create('owner', 'super');
		
		if($form->valid)
		{
			\model\User::authenticate($user->id);
			
			return;
		}
		
		\lib\Template::render('pawn/setup.html', array
		(
			'heading'		=>	Lang::get('heading_setup'),
			'about'			=>	Lang::get('super_user'),
			'sidebar_about'	=>	Lang::get('setup_process'),
			
			'form'		=>	$form
		));
		
		exit;
	}
	
	# Construct
	public function _construct()
	{
		/*
		if(!$this->user)
		{
			# HTTP Authentication
			if(isset($_SERVER['PHP_AUTH_USER']))
			{
				$this->user = \model\User::authenticate(utf8_decode($_SERVER['PHP_AUTH_USER']), utf8_decode($_SERVER['PHP_AUTH_PW']));
			}
			
			if(!$this->user)
			{
				header('WWW-Authenticate: Basic');
				header('HTTP/1.0 401 Unauthorized');
				
				die('<DOCTYPE html><html><head><title>Unauthorized</title><meta http-equiv="refresh" content="1;'.WEB.'"/></head><body><p>You are not authorized to access this page.</p></body></html>');
			}
		}
		*/
		if($this->user->level < 4)
		{
			header('Location: '.WEB);
			
			exit;
		}
	}
	
	# Navigation
	public function navigation($current)
	{
		$links = ($this->user->level == 5)?array('settings'):array();
		
		if($this->user->level == 5 && file_exists(ROOT.'config/advertisements.yml'))
		{
			$links[] = 'advertisements';
		}
		
		$navigation = array(/*'Home'=>1, */'Statistics'=>1);
		
		foreach($links as $link)
		{
			$navigation[$link] = array
			(
				'name'		=>	Lang::get($link),
				'path'		=>	\Pawn::path('Pawn', $link),
				'current'	=>	$current == $link
			);
		}
		
		$models = db::execute('SELECT model FROM '.PREFIX.'migration')->fetchAll(\PDO::FETCH_COLUMN);
		
		foreach($models as $model)
		{
			if(Lang::get('admin', $model))
			{
				$navigation[$model] = array
				(
					'name'		=>	Lang::get('admin', $model, 'instances'),
					'path'		=>	\Pawn::path('Pawn', 'model', array('name'=>$model)),
					'current'	=>	$current == $model
				);
			}
		}
		
		$trash = $navigation['Trash'];
		
		unset($navigation['Trash'], $navigation['Statistics'], $navigation['Home']);
		
		$navigation['Trash'] = $trash;
		
		return $navigation;
	}
	
	# Model
	const pageSize = 15;
	
	public function model($name='Home')
	{
		if($name == 'Home')
		{
			$this->redirect('Pawn', 'settings');
		}
		
		if(!Lang::get('admin', $name))
		{
			$this->redirect('Pawn', 'home');
		}
		
		$class = '\\model\\'.$name;
		
		// Custom
		if(method_exists($class, 'model_view'))
		{
			return call_user_func(array($class, 'model_view'), $this);
		}
		
		// Listing
		$head = Lang::get('admin', $name, 'listing');
		$total = call_user_func(array($class, 'num_rows'));
		
		if(@constant($class.'::sortColumn') !== null)
		{
			$sortColumn = constant($class.'::sortColumn');
		} else
		{
			$keys = array_keys($head);
			
			$sortColumn = $keys[0];
		}

		$sortAscending = (@constant($class.'::sortAscending') !== null)?constant($class.'::sortAscending'):1;
		
		// Actions
		$actions = method_exists($class, 'actions')?call_user_func(array($class, 'actions')):array();
		
		// Vars
		$vars = array
		(
			'heading'		=>	Lang::get('admin', $name, 'instances'),
			'about'			=>	Lang::get('model_total', $total),
			
			'name'			=>	$name,
			'subheading'	=>	Lang::get('admin', $name, 'instances'),
			
			'head'			=>	$head,
			'total'			=>	$total,
			'data'			=>	$this->rows($name, 0, $sortColumn, $sortAscending, false),
			
			'actions'		=>	$actions,
			'bulk'			=>	method_exists($class, 'bulk')?call_user_func(array($class, 'bulk')):array(),
			
			'navigation'	=>	$this->navigation($name),
			
			'pageSize'		=>	self::pageSize,
			
			'sortColumn'	=>	$sortColumn,
			'sortAscending'	=>	$sortAscending
		);
		
		// Render
		if(method_exists($class, 'model_vars'))
		{
			$vars = call_user_func(array($class, 'model_vars'), $vars);
		}
		
		return $this->render('pawn/model.html', $vars);
	}
	
	# Rows
	public function rows($name, $page, $sort, $asc, $output=true)
	{
		if(!Lang::get('admin', $name))
		{
			exit;
		}
		
		$class = '\\model\\'.$name;
		
		$page = (int) $page;
		$pageSize = self::pageSize;
		
		$asc = ((bool) $asc)?'ASC':'DESC';
		
		$listing = array_keys(Lang::get('admin', $name, 'listing'));
		
		if(isset($_POST['search']))
		{
			$query = 'WHERE `'.$listing[0].'` LIKE '.db::quote($_POST['search'].'%').' ';
		} else
		{
			$query = null;
		}
		
		$total = call_user_func(array($class, 'num_rows'), $query);
		
		$query .= "ORDER BY `$sort` $asc LIMIT ".($page*$pageSize).",$pageSize";
		
		$data = array();
		$rows = call_user_func(array($class, 'get'), $query)->fetchAll();
		
		foreach($rows as $instance)
		{
			$row = array($instance->id);
			
			$values = $instance->listing();
			
			foreach($listing as $key)
			{
				$row[] = is_array($values[$key])?$values[$key][0]:nl2br(str_replace(array('<', '>'), array('&lt;', '&gt;'), $values[$key]));
			}
			
			$data[] = $row;
		}
		
		return $this->json(array
		(
			'rows'	=>	$data,
			'total'	=>	$total
		), $output);
	}
	
	# Action
	public function action($action, $model)
	{
		$class = '\\model\\'.$model;
		
		if(!method_exists($class, $action))
		{
			$this->redirect('Pawn', 'model', array('name'=>$model));
		}
		
		return call_user_func(array($class, $action), $this);
	}
	
	# Bulk
	public function bulk($action, $model)
	{
		$class = '\\model\\'.$model;
		
		$bulk = call_user_func(array($class, 'bulk'));
		
		if(!isset($bulk[$action]))
		{
			exit;
		}
		
		if(is_string($bulk[$action]))
		{
			$action = $bulk[$action];
		}
		
		if($action == 'edit')
		{
			return $this->json(array
			(
				'redirect'	=>	\Pawn::path('Pawn', 'edit', array
				(
					'model'		=>	$model,
					'instances'	=>	$_POST['instances']
				))
			));
		}
		
		$instances = explode(',', $_POST['instances']);
		
		rsort($instances);
		
		foreach($instances as $id)
		{
			if($instance = call_user_func(array($class, 'get'), 'WHERE id=?', $id, false)->fetch())
			{
				$instance->$action();
			}
		}
		
		return $this->json(array
		(
			'redraw'	=>	true
		));
	}
	
	# Add
	public function add($name)
	{
		$class = '\\model\\'.$name;
		
		$row = new $class;
		
		$form = $row->create('instance', 'administrate');
		
		if($form->valid)
		{
			$this->redirect('Pawn', 'model', array('name'=>$name));
		}
		
		return $this->render('pawn/add.html', array
		(
			'heading'		=>	Lang::get('admin', $name, 'instances'),
			'about'			=>	Lang::get('add_instance'),
			
			'name'			=>	$name,
			
			'subheading'	=>	Lang::get('admin', $name, 'add'),
			'form'			=>	$form,
			
			'navigation'	=>	$this->navigation($name)
		));
	}
	
	# Edit
	public function edit($model, $instances)
	{
		$forms = array();
		$class = '\\model\\'.$model;
		
		$instances = explode(',', $instances);
		
		foreach($instances as $index=>$id)
		{
			if($row = call_user_func(array($class, 'get'), 'WHERE id=?', $id)->fetch())
			{
				if($model == 'Trash')
				{
					$row = call_user_func(array('\\model\\'.$row->model, 'get'), 'WHERE id=?', $row->model_id, 1)->fetch();
				}
				
				$form = $row->edit('instance', 'administrate');
				
				if($form->valid)
				{
					unset($instances[$index]);
				} else
				{
					$listing = $row->listing();
					
					$forms[$row->__toString()] = $form;
				}
			} else
			{
				unset($instances[$index]);
			}
		}
		
		if(!count($instances))
		{
			$this->redirect('Pawn', 'model', array('name'=>$model));
		}
		
		return $this->render('pawn/edit.html', array
		(
			'heading'		=>	Lang::get('admin', $model, 'instances'),
			'about'			=>	Lang::get('edit_instances'),
			
			'name'			=>	$model,
			
			'forms'			=>	$forms,
			'instances'		=>	implode(',', $instances),
			
			'navigation'	=>	$this->navigation($model)
		));
	}
	
	# Settings
	public function settings()
	{
		if($this->user->level < 5)
		{
			header('Location: '.\Pawn::path('Pawn', 'model', array('name'=>'Game')));
			
			exit;
		}
		
		// Forms
		$forms = array();
		
		// Page settings
		$form = new \lib\Form('settings_form');
		
		$form->h2 = Lang::get('page_settings');
		
		$form->field('title', 'text', array(), \Pawn::setting('title'));
		$form->field('company', 'text', array('optional'=>true), \Pawn::setting('company'));
		
		// Locale
		$locales = array();
		
		$directory = opendir(ROOT.'lang/');
		
		while(($file = readdir($directory)) !== false)
		{
			if($file == '.' || $file == '..') continue;
			
			$language = \Pawn::config('lang/'.$file.'/lang.yml');
			
			$locales[$file] = $language['name'];
		}
		
		if(count($locales) > 1)
		{
			$form->field('locale', 'select', $locales, \Pawn::setting('locale'));
		}
		
		// Timezone
		$form->field('timezone', 'select', timezone_identifiers_list(), \Pawn::setting('timezone'));
		
		date_default_timezone_set($form->timezone);
		
		// Date format
		$time = time();
		
		$date_formats = array
		(
			'M j, Y'	=>	date('M j, Y', $time),
			'd.m.Y'		=>	date('d.m.Y', $time),
			'm.d.Y'		=>	date('m.d.Y', $time)
		);
		
		$form->field('date_format', 'select', $date_formats, \Pawn::setting('date_format'));
		
		// Template
		$templates = array();
		
		$directory = opendir(ROOT.'public/template/');
		
		while(($file = readdir($directory)) !== false)
		{
			if($file == '.' || $file == '..') continue;
			
			$template = \Pawn::config('public/template/'.$file.'/theme.yml');
			
			$templates[$file] = $template['name'];
		}
		
		if(count($templates) > 1)
		{
			$form->field('template', 'select', $templates, \Pawn::setting('template'));
		}
		
		// Save settings
		if($form->validate())
		{
			if(isset($form->template) && $form->fields['template'] != \Pawn::setting('template'))
			{
				$public_cache = glob(ROOT.'cache/public_*');
				
				foreach($public_cache as $file)
				{
					unlink($file);
				}
			}
			
			foreach($form->fields as $key=>$value)
			{
				\Pawn::setting($key, $value);
				
				db::query('UPDATE `'.PREFIX.'config` SET `value`=? WHERE `key`=?', array($value, $key));
			}
		}
		
		$forms[] = $form;
		
		// Models
		$models = db::execute('SELECT model FROM '.PREFIX.'migration')->fetchAll(\PDO::FETCH_COLUMN);
		
		foreach($models as $model)
		{
			if(method_exists('\\model\\'.$model, 'config'))
			{
				$form = new Form('settings_form');
				
				$form->h2 = Lang::get('admin', $model, 'settings');
				$form->instructions = Lang::get('admin', $model, 'settings_instructions');
				
				$config = call_user_func(array('\\model\\'.$model, 'config'));
				
				foreach($config as $name=>$field)
				{
					$field[2] = \Pawn::setting($model.'.'.$name);
					
					call_user_func_array(array($form, 'field'), array_merge((array) $name, $field));
				}
				
				if($form->validate())
				{
					foreach($config as $name=>$field)
					{
						\Pawn::setting($model.'.'.$field[0], $form->request($name));
						
						db::query('UPDATE `'.PREFIX.'config` SET `value`=? WHERE `key`=?', array($form->request($name), $model.'.'.$name));
					}
				}
				
				$forms[] = $form;
			}
		}
		
		// View
		return $this->render('pawn/settings.html', array
		(
			'heading'		=>	Lang::get('settings'),
			'about'			=>	Lang::get('settings_about'),
			
			'forms'			=>	$forms,
			
			'navigation'	=>	$this->navigation('settings')
		));
	}
	
	# Advertisements
	public function advertisements()
	{
		if($this->user->level < 5)
		{
			exit;
		}
		
		$sizes = array();
		
		$config = \Pawn::config('config/advertisements.yml');
		
		if(isset($_POST['advertisements']))
		{
			foreach($config as $size)
			{
				$key = 'ad_'.$size;
				$value = $_POST[$size];
				
				if(\Pawn::setting($key) === null)
				{
					db::query('INSERT INTO `'.PREFIX.'config` VALUES (?, ?)', array($key, $value));
				} else
				{
					db::query('UPDATE `'.PREFIX.'config` SET `value`=? WHERE `key`=?', array($value, $key));
				}
				
				\Pawn::setting($key, $value);
			}
		}
		
		foreach($config as $size)
		{
			$sizes[$size] = \Pawn::setting('ad_'.$size);
		}
		
		return $this->render('pawn/advertisements.html', array
		(
			'heading'		=>	Lang::get('advertisements'),
			'about'			=>	Lang::get('advertisements_about'),
			
			'sizes'			=>	$sizes,
			
			'navigation'	=>	$this->navigation('advertisements')
		));
	}
	
	/*
	# Log out
	public function log_out()
	{
		$this->user = $this->user->unauthenticate();
		
		header('HTTP/1.0 401 Unauthorized');
		
		die('<DOCTYPE html><html><head><title>Unauthorized</title><meta http-equiv="refresh" content="1;'.WEB.'"/></head><body><p>You are not authorized to access this page.</p></body></html>');
	}
	*/
}